#include <Carbon.h>
#include <string.h>
#include <stdio.h>

#include "keyboard.h"
#include "gworld.h"
#include "stringtools.h"
#include "dialog.h"
#include "main.h"
#include "keyselect.h"
#include "graphics.h"

GWorldPtr keyboardWorld, keyBuffer, keyDrawBuffer, keyBackBuffer;
Handle keyIcon[kKeys];
Rect keyboardRect;
Rect keyRect[0x80][2];

#define kKeyCodeA   0x00
#define kKeyCodeD   0x02
#define kKeyCodeX   0x07
#define kKeyCodeS   0x01
#define kKeyCodeLt  0x7b
#define kKeyCodeRt  0x7c
#define kKeyCodeDn  0x7d
#define kKeyCodeUp  0x7e
#define kKeyCodeEsc 0x35

UInt8 defaultKeys[kKeys];
UInt8 keyCode[kKeys] = 
{
	kKeyCodeA,
	kKeyCodeD,
	kKeyCodeX,
	kKeyCodeS,
	kKeyCodeLt,
	kKeyCodeRt,
	kKeyCodeDn,
	kKeyCodeUp,
	kKeyCodeEsc
};

#define kIconSize 16
#define kKeySize 24
#define KS kKeySize

Rect keyboardSize = {0, 0, kKeySize * 7 + 1, kKeySize * 23 + 1 };

static KeyboardLayout keys[] =
{
	{ KS, KS, 0x35, "esc" },
	{ KS, KS, 0x00, nil },
	{ KS, KS, 0x7A, "F1" },
	{ KS, KS, 0x78, "F2" },
	{ KS, KS, 0x63, "F3" },
	{ KS, KS, 0x76, "F4" },
	{ KS/2, KS, 0x00, nil },
	{ KS, KS, 0x60, "F5" },
	{ KS, KS, 0x61, "F6" },
	{ KS, KS, 0x62, "F7" },
	{ KS, KS, 0x64, "F8" },
	{ KS/2, KS, 0x00, nil },
	{ KS, KS, 0x65, "F9" },
	{ KS, KS, 0x6d, "F10" },
	{ KS, KS, 0x67, "F11" },
	{ KS, KS, 0x6f, "F12" },
	{ KS/2, KS, 0x00, nil },
	{ KS, KS, 0x69, "F13" },
	{ KS, KS, 0x6b, "F14" },
	{ KS, KS, 0x71, "F15" },
	{ 0,  0,  0x00, nil },

	{ 0,  0,  0x00, nil },

	{ KS, KS, 0x32, "`" },
	{ KS, KS, 0x12, "1" },
	{ KS, KS, 0x13, "2" },
	{ KS, KS, 0x14, "3" },
	{ KS, KS, 0x15, "4" },
	{ KS, KS, 0x17, "5" },
	{ KS, KS, 0x16, "6" },
	{ KS, KS, 0x1a, "7" },
	{ KS, KS, 0x1c, "8" },
	{ KS, KS, 0x19, "9" },
	{ KS, KS, 0x1d, "0" },
	{ KS, KS, 0x1b, "-" },
	{ KS, KS, 0x18, "=" },
	{ KS*2, KS, 0x33, "delete" },
	{ KS/2, KS, 0, nil },
	{ KS, KS, 0x72, "ins" },
	{ KS, KS, 0x73, "hom" },
	{ KS, KS, 0x74, "pgu" },
	{ KS/2, KS, 0, nil },
	{ KS, KS, 0x47, "clr" },
	{ KS, KS, 0x51, "=" },
	{ KS, KS, 0x4b, "/" },
	{ KS, KS, 0x43, "*" },
	{ 0,  0,  0x00, nil },

	{ KS*3/2, KS, 0x30, "tab" },
	{ KS, KS, 0x0c, "Q" },
	{ KS, KS, 0x0d, "W" },
	{ KS, KS, 0x0e, "E" },
	{ KS, KS, 0x0f, "R" },
	{ KS, KS, 0x11, "T" },
	{ KS, KS, 0x10, "Y" },
	{ KS, KS, 0x20, "U" },
	{ KS, KS, 0x22, "I" },
	{ KS, KS, 0x1f, "O" },
	{ KS, KS, 0x23, "P" },
	{ KS, KS, 0x21, "[" },
	{ KS, KS, 0x1e, "]" },
	{ KS*3/2, KS, 0x2a, "\\" },
	{ KS/2, KS, 0, nil },
	{ KS, KS, 0x75, "del" },
	{ KS, KS, 0x77, "end" },
	{ KS, KS, 0x79, "pgd" },
	{ KS/2, KS, 0, nil },
	{ KS, KS, 0x59, "7" },
	{ KS, KS, 0x5b, "8" },
	{ KS, KS, 0x5c, "9" },
	{ KS, KS, 0x4e, "-" },
	{ 0,  0,  0x00, nil },
	
	{ KS*2, KS, 0x39, "caps" },
	{ KS, KS, 0x00, "A" },
	{ KS, KS, 0x01, "S" },
	{ KS, KS, 0x02, "D" },
	{ KS, KS, 0x03, "F" },
	{ KS, KS, 0x05, "G" },
	{ KS, KS, 0x04, "H" },
	{ KS, KS, 0x26, "J" },
	{ KS, KS, 0x28, "K" },
	{ KS, KS, 0x25, "L" },
	{ KS, KS, 0x29, ";" },
	{ KS, KS, 0x27, "" },
	{ KS*2, KS, 0x24, "return" },
	{ KS*4, KS, 0, nil },
	{ KS, KS, 0x56, "4" },
	{ KS, KS, 0x57, "5" },
	{ KS, KS, 0x58, "6" },
	{ KS, KS, 0x45, "+" },
	{ 0,  0,  0x00, nil },
	
	{ KS*5/2, KS, 0x38, "shift" },
	{ KS, KS, 0x06, "Z" },
	{ KS, KS, 0x07, "X" },
	{ KS, KS, 0x08, "C" },
	{ KS, KS, 0x09, "V" },
	{ KS, KS, 0x0b, "B" },
	{ KS, KS, 0x2d, "N" },
	{ KS, KS, 0x2e, "M" },
	{ KS, KS, 0x2b, "," },
	{ KS, KS, 0x2f, "." },
	{ KS, KS, 0x2c, "/" },
	{ KS*5/2, KS, 0x38, "shift" },
	{ KS*3/2, KS, 0, nil },
	{ KS, KS, 0x7e, "up" },
	{ KS*3/2, KS, 0, nil },
	{ KS, KS, 0x53, "1" },
	{ KS, KS, 0x54, "2" },
	{ KS, KS, 0x55, "3" },
	{ KS, KS*2, 0x4c, "ent" },
	{ 0,  0,  0x00, nil },
	
	{ KS*3/2, KS, 0x3b, "ctrl" },
	{ KS*3/2, KS, 0x3a, "opt" },
	{ KS*3/2, KS, 0x37, "cmd" },
	{ KS*6, KS, 0x31, "" },
	{ KS*3/2, KS, 0x37, "cmd" },
	{ KS*3/2, KS, 0x3a, "opt" },
	{ KS*3/2, KS, 0x3b, "ctrl" },
	{ KS/2, KS, 0, nil },
	{ KS, KS, 0x7b, "lt" },
	{ KS, KS, 0x7d, "dn" },
	{ KS, KS, 0x7c, "rt" },
	{ KS/2, KS, 0, nil },
	{ KS*2, KS, 0x52, "0" },
	{ KS, KS, 0x41, "." },
	{ 0,  0,  0x00, nil }
};

enum
{
	iOK = 1,
	iDefaults,
	iJoystick,
	iKeyboard
};

/* Starts the keyboard dialog and runs it with ModalDialog. */

void KeyboardSetup( void )
{
	Rect keyBufferRect = {0, 0, kIconSize+2, kIconSize+2};
	int index;
	DialogItemIndex item;
	Rect frontDialogRect;
	
	/* Pause DSp */
	if( context )
	{
		DSpContext_SetState( context, kDSpContextState_Paused );
	}

	/* Load key icons into an array. 
	   Note: can't do this in InitKeyboard because opening the InputSprocket
	   dialog will release these icon suites without our permission. Ouch. 
	   Icon Services' refcounting doesn't seem to help this either.  */	

	for( index=0; index<kKeys; index++ )
	{
		GetIconSuite( &keyIcon[index], 800 + index, svAllSmallData );
	}

	StartDialog( dKeySelect, KeyboardSetupFilterProc );
	SetDialogDefaultItem( frontDialog, iOK );
	GetDialogRect( frontDialog, iKeyboard, &keyboardRect );
	keyboardRect.bottom = keyboardRect.top + keyboardSize.bottom;
	keyboardRect.right = keyboardRect.left + keyboardSize.right;
	
	GetPortBounds( GetWindowPort( GetDialogWindow( frontDialog ) ), &frontDialogRect );
	InitGWorld( &keyBuffer,     &keyBufferRect,   16 );
	InitGWorld( &keyDrawBuffer, &keyBufferRect,   16 );
	InitGWorld( &keyBackBuffer, &frontDialogRect, 16 );

	ShowWindow( GetDialogWindow(frontDialog) );
	SetPortDialogPort( frontDialog );		

	FlushEvents( everyEvent, 0L );
	ArrowCursor( );
	
	do
	{
		ModalDialog( DialogFilter, &item );

		switch( item )
		{
			case iDefaults:
				memcpy( keyCode, defaultKeys, sizeof(keyCode) );
				SetPortDialogPort( frontDialog );
				InvalWindowRect( GetDialogWindow(frontDialog), &keyboardRect );
				break;
			
			case iJoystick:
				ConfigureInputSprocket( );

				// The InputSprocket dialog will leave a nasty hole 
				// in the background windows, and may even dispose
				// of our Keyboard icons, dammit. Fix this.
				
				RefreshAll();
				for( index=0; index<kKeys; index++ )
				{
					GetIconSuite( &keyIcon[index], 800 + index, svAllSmallData );
				}
				break;
		}

	}
	while( item != iOK );
	
	DisposeGWorld( keyBuffer     );
	DisposeGWorld( keyDrawBuffer );
	DisposeGWorld( keyBackBuffer );
	FinishDialog( 800 );

	/* Unpause DSp */
	if( context )
	{
		DSpContext_SetState( context, kDSpContextState_Active );
		DSpContext_GetFrontBuffer( context, &backdropPort );
	}
}


/* Initialize the keyboard function. */

void InitKeyboard( void )
{
	short row, index, scancode;
	Rect myRect;
	
	/* Set the default keys to the current keycodes; since InitKeyboard is
	   called before preferences are loaded, this is cool. */
	   
	memcpy( defaultKeys, keyCode, sizeof(keyCode) );

	/* Clear the key rects. */
	
	for( index=0; index<0x80; index++ )
	{
		keyRect[index][0].top    = keyRect[index][0].left  =
		keyRect[index][0].bottom = keyRect[index][0].right = 
		keyRect[index][1].top    = keyRect[index][1].left  =
		keyRect[index][1].bottom = keyRect[index][1].right = 0;
	}

	/* Set up the GWorld containing the keyboard picture. */	
	InitGWorld( &keyboardWorld, &keyboardSize, 16 );
	PrepareForGDrawing( keyboardWorld );
	SetThemePen( kThemeBrushModelessDialogBackgroundActive, 16, true );
	PaintRect( &keyboardSize );
	SetThemePen( kThemeBrushBlack, 16, true );
	ForeColor( blackColor );
	BackColor( whiteColor );
	
	/* Draw the keyboard, row by row. */
	TextFont( kFontIDGeneva );
	TextSize( 10 );
	
	myRect.top = myRect.left = myRect.right = myRect.bottom = 0;
	
	index = 0;

	for( row=0; row<7; row++ )
	{	
		while( keys[index].keyWidth )
		{
			myRect.right += keys[index].keyWidth;
			
			if( keys[index].keyLabel )
			{
				myRect.bottom = myRect.top + keys[index].keyHeight + 1;
				
				scancode = keys[index].scancode;
				
				if( keyRect[scancode][0].bottom == 0 )
					keyRect[scancode][0] = myRect;
				else
					keyRect[scancode][1] = myRect;
				
				RenderKey( &keys[index], &myRect );
			}
			
			myRect.left = myRect.right;
			
			index++;
		}
		
		index++;
		OffsetRect( &myRect, 0, kKeySize );
		myRect.left = myRect.right = 0;
	}
	/*
	{
		FILE *f;
		int count;
		f=fopen("keyboard.txt","w");
		for( count=0; count<128; count++ )
		{
			index = 0;
			for( row=0; row<7; row++ )
			{
				while( keys[index].keyWidth )
				{
					if( keys[index].scancode == count )
					{
						fprintf(f, "\"%s\",\n", keys[index].keyLabel );
						goto next;
					}					
					index++;
				}
				index++;
			}
			fprintf(f, "\"??\",\n", keys[index].keyLabel );
			next: ;
		}
	}
	*/
	FinishGDrawing( keyboardWorld );
}

/* Draw an individual key cap. */

void RenderKey( KeyboardLayout *key, Rect *rect )
{
	short textInset = 4;
	
	rect->right++;
	
	DrawShadedBox( rect );
	
	if( (key->scancode >= 0x67 && key->scancode <= 0x71)  ) // if between F10 and F15
	{
		textInset -= 2;
	}
	
	if( (key->scancode == 0x77) || (key->scancode == 0x74) || (key->scancode == 0x79) ) // end/pgu/pgd
	{
		textInset -= 2;//3;
	}
	
	if( (key->scancode == 0x73) ) // hom
	{
		textInset -= 3;//4;	
	}
	
	MoveTo( rect->left + textInset, rect->bottom - 7 );
	ForeColor( blackColor );
	DrawText( key->keyLabel, 0, strlen( key->keyLabel ) );
		
	rect->right--;
}

/* Plot the key icon(s) on top of the key caps. */

void PlotKeyIcon( short which )
{
	RGBColor foreColor, backColor, grayColor = {0x8000, 0x8000, 0x8000};
	Rect worldRect, myRect;
	short each;
	GrafPtr thePort;
	
	GetForeColor( &foreColor );
	GetBackColor( &backColor );
	
	ForeColor( blackColor );
	BackColor( whiteColor );
	
	for( each=0; each<=1; each++ )
	{
		worldRect = keyRect[keyCode[which]][each];
		worldRect.bottom--;
		myRect = worldRect;
		
		if( myRect.bottom > 0 )
		{
			OffsetRect( &myRect, keyboardRect.left, keyboardRect.top );
			
			PaintRect( &myRect );
			OpColor( &grayColor );
			GetPort( &thePort );
			CopyBits( GetPortBitMapForCopyBits(keyboardWorld), 
					  GetPortBitMapForCopyBits(thePort),
					  &worldRect, &myRect,
					  blend, nil );

			myRect.left += ( myRect.right - myRect.left - kIconSize ) / 2;
			myRect.right = myRect.left + kIconSize;
			myRect.top  += ( myRect.bottom - myRect.top - kIconSize ) / 2;
			myRect.bottom = myRect.top + kIconSize;
			
			PlotIconSuite( &myRect, atAbsoluteCenter, ttNone, keyIcon[which] );
	
			InsetRect( &myRect, -1, -1 );
			FrameRect( &myRect );
		}
	}
	
	RGBForeColor( &foreColor );
	RGBBackColor( &backColor );
}

/* Erase the icons on top of key caps. */

void UnplotKeyIcon( short which )
{
	RGBColor foreColor, backColor;
	Rect dialogRect, worldRect;
	short each;
	GrafPtr thePort;
	 
	GetForeColor( &foreColor );
	GetBackColor( &backColor );
	
	ForeColor( blackColor );
	BackColor( whiteColor );

	for( each=0; each<=1; each++ )
	{
		worldRect = keyRect[keyCode[which]][each];
		worldRect.bottom--;
		dialogRect = worldRect;
		
		if( worldRect.bottom > 0 )
		{
			OffsetRect( &dialogRect, keyboardRect.left, keyboardRect.top );
			
			GetPort( &thePort );
			CopyBits( GetPortBitMapForCopyBits(keyboardWorld), 
					  GetPortBitMapForCopyBits(thePort),
					  &worldRect, &dialogRect,
					  srcCopy, nil );
		}
	}
	
	RGBForeColor( &foreColor );
	RGBBackColor( &backColor );
}

/* Let the user drag the key rects around. */

short DragGrayRect( Point mouseAt, short changeKey )
{
	RGBColor foreColor, backColor, grayColor = {0x8000, 0x8000, 0x8000};
	Rect oldRect, drawRect, bufferRect = {0, 0, kIconSize+2, kIconSize+2};
	Point newMouse;
	short which, each;
	Rect portRect;
	
	newMouse.h = mouseAt.h - keyboardRect.left;
	newMouse.v = mouseAt.v - keyboardRect.top;
	
	if( PtInRect( newMouse, &keyRect[ keyCode[ changeKey ] ][0] ) )
		drawRect = keyRect[ keyCode[ changeKey ] ][0];
	else
		drawRect = keyRect[ keyCode[ changeKey ] ][1];
	
	drawRect.left += ( drawRect.right - drawRect.left - kIconSize ) / 2;
	drawRect.right = drawRect.left + kIconSize;
	drawRect.top  += ( drawRect.bottom - drawRect.top - kIconSize ) / 2;
	drawRect.bottom = drawRect.top + kIconSize;

	OffsetRect( &drawRect, keyboardRect.left, keyboardRect.top );
	
	InsetRect( &drawRect, -1, -1 );
	
	SetPortDialogPort( frontDialog );

	GetForeColor( &foreColor );
	GetBackColor( &backColor );
	
	ForeColor( blackColor );
	BackColor( whiteColor );
	
	SetPortDialogPort( frontDialog );

	CopyBits( GetPortBitMapForCopyBits( GetDialogPort( frontDialog ) ),  GetPortBitMapForCopyBits( keyBuffer ),
	          &drawRect, &bufferRect,
	          srcCopy, nil );
	
	GetPortBounds( GetDialogPort( frontDialog ), &portRect );
	CopyBits( GetPortBitMapForCopyBits( GetDialogPort( frontDialog ) ),  GetPortBitMapForCopyBits( keyBackBuffer ),
	          &portRect, &portRect,
	          srcCopy, nil );

	PrepareForGDrawing( keyDrawBuffer );
	OpColor( &grayColor );
	FinishGDrawing( keyDrawBuffer );
	
	SetPortDialogPort( frontDialog );
	
	if( hasAppearance11 ) BeginThemeDragSound( kThemeDragSoundDragging );

	while( StillDown( ) )
	{
		GetMouse( &newMouse );
		if( newMouse.h != mouseAt.h || newMouse.v != mouseAt.v )
		{
			oldRect = drawRect;
			OffsetRect( &drawRect, newMouse.h - mouseAt.h, newMouse.v - mouseAt.v );
			mouseAt = newMouse;
						
			PrepareForGDrawing( keyDrawBuffer );
			CopyBits( GetPortBitMapForCopyBits( keyBackBuffer ), GetPortBitMapForCopyBits( keyDrawBuffer ),
			          &drawRect, &bufferRect,
			          srcCopy, nil );
			CopyBits( GetPortBitMapForCopyBits( keyBuffer ), GetPortBitMapForCopyBits( keyDrawBuffer ),
			          &bufferRect, &bufferRect,
			          blend, nil );
			FinishGDrawing( keyDrawBuffer ); 
			
			SetPortDialogPort( frontDialog );
			CopyBits( GetPortBitMapForCopyBits( keyBackBuffer ), GetPortBitMapForCopyBits( GetDialogPort( frontDialog ) ),
			          &oldRect, &oldRect,
			          srcCopy, nil );
			CopyBits( GetPortBitMapForCopyBits( keyDrawBuffer ), GetPortBitMapForCopyBits( GetDialogPort( frontDialog ) ),
			          &bufferRect, &drawRect, 
			          srcCopy, nil );
		}
	}

	CopyBits( GetPortBitMapForCopyBits( keyBackBuffer ),  GetPortBitMapForCopyBits( GetDialogPort( frontDialog ) ),
	          &drawRect, &drawRect, 
	          srcCopy, nil );

	if( hasAppearance11 ) EndThemeDragSound(  );
	
	RGBForeColor( &foreColor );
	RGBBackColor( &backColor );

	mouseAt.h -= keyboardRect.left;
	mouseAt.v -= keyboardRect.top;
	
	for( which=0; which<0x80; which++ )
	{
		for( each=0; each<=1; each++ )
		{
			if( PtInRect( mouseAt, &keyRect[which][each] ) && !KeyCodeInUse( which ) )
			{
				if( hasAppearance11 ) PlayThemeSound( kThemeSoundReceiveDrop );
				return which;
			}
		}
	}
	
	return -1;
}

/* Checks whether a key is currently assigned to anything or not. */

Boolean KeyCodeInUse( short which )
{
	short count;
	
	for( count=0; count<kKeys; count++ )
	{
		if( keyCode[count] == which ) return true;
	}
	return false;
}

/* Given a point in the dialog, determines which key code is represented. */

short FindHitKey( Point where )
{
	short hit, which, each;
	
	where.v -= keyboardRect.top;
	where.h -= keyboardRect.left;

	hit = -1;
	for( which = 0; which < kKeys; which++ )
	{
		for( each=0; each<=1; each++ )
		{
			if( PtInRect( where, &keyRect[ keyCode[ which ] ][each] ) )
			{
				hit = which;
			}
		}
	}
	
	return hit;
}

/* Filter proc for the keyboard setup dialog.
   Note: This would be much cleaner as a user item procedure.
   But this code was written pre-Appearance, and it still works
   great, so I haven't found the need to update it yet... */
    
pascal Boolean KeyboardSetupFilterProc( DialogPtr theDialog,
										EventRecord *theEvent,
										short *itemHit )
{
	static RgnHandle tempRgn;
	if( tempRgn == NULL ) tempRgn = NewRgn();
	
	if( theEvent->what == updateEvt )
	{
		RGBColor foreColor, backColor;
		Rect portRect;
		short which;
		
		SetPortDialogPort( theDialog );
			
		GetForeColor( &foreColor );
		GetBackColor( &backColor );
		
		UpdateDialog( theDialog, GetPortVisibleRegion( GetDialogPort( theDialog ), tempRgn ) );
				
		ForeColor( blackColor );
		BackColor( whiteColor );
		GetPortBounds( keyboardWorld, &portRect );
		CopyBits( GetPortBitMapForCopyBits( keyboardWorld ), GetPortBitMapForCopyBits( GetDialogPort( theDialog ) ),
				  &portRect, &keyboardRect,
				  srcCopy, nil );
		
		for( which=0; which<kKeys; which++ )
		{
			PlotKeyIcon( which );
		}
		
		RGBForeColor( &foreColor );
		RGBBackColor( &backColor );
		
		BeginUpdate( (WindowPtr) theEvent->message );
		EndUpdate( (WindowPtr) theEvent->message );
	}

	if( theEvent->what == mouseDown )
	{
		Point localHit = theEvent->where;
		short changeKey, changeTo;

		GlobalToLocal( &localHit );

		if( PtInRect( localHit, &keyboardRect ) )
		{
			changeKey = FindHitKey( localHit );
			if( changeKey != -1 )
			{
				changeTo = DragGrayRect( localHit, changeKey );
				
				if( changeTo != -1 )
				{
					UnplotKeyIcon( changeKey );
				 	keyCode[changeKey] = changeTo;
					PlotKeyIcon( changeKey );
				}
			}
		}
	}
	
	if( theEvent->what == keyDown )
	{
		switch ( theEvent->message & charCodeMask )
		{
			case '\r':
			case 0x3:
				*itemHit = ok;
				FlashItem( theDialog, ok );
				return true;
		}
	}
	
	return false;
}

void DrawShadedBox( Rect *rect )
{
	if( hasAppearance11 )
	{
		ThemeButtonDrawInfo info = { kThemeStateActive, kThemeButtonOff, kThemeAdornmentNone };
		rect->right--; rect->bottom--;
		DrawThemeButton( rect, kThemeMediumBevelButton, &info, NULL, NULL, NULL, 0L );
		rect->right++; rect->bottom++;
	}
	else
	{
		RGBColor ltGray = { 0xFFFF, 0xFFFF, 0xFFFF },
				 mdGray = { 0xDDDD, 0xDDDD, 0xDDDD },
				 dkGray = { 0xAAAA, 0xAAAA, 0xAAAA },
				 dpGray = { 0x7777, 0x7777, 0x7777 };

		ForeColor( blackColor );
		FrameRect( rect );
		
		InsetRect( rect, 1, 1 );
		RGBForeColor( &mdGray );
		PaintRect( rect );
		
		RGBForeColor( &ltGray );
		MoveTo( rect->right-2, rect->top+1 );
		LineTo( rect->left+1, rect->top+1 );
		LineTo( rect->left+1, rect->bottom-2 );
		
		RGBForeColor( &dpGray );
		MoveTo( rect->right-1, rect->top+1 );
		LineTo( rect->right-1, rect->bottom-1 );
		LineTo( rect->left+1, rect->bottom-1 );
		
		RGBForeColor( &dkGray );
		MoveTo( rect->left+2, rect->bottom-2 );
		LineTo( rect->right-2, rect->bottom-2 );
		LineTo( rect->right-2, rect->top+2 );
		
		InsetRect( rect, -1, -1 );
	}
}
